//-----------------------------------------------------------------------------
//Copyright  2000-2003 TogetherSoft Corporation, a wholly owned subsidiary of Borland Software Corporation. All rights reserved. All patents pending.
//-----------------------------------------------------------------------------
namespace CSCashSales.data_objects
{
    using System;

	/// <summary>
	/// This is the class used to hold the description of a sale position.
	/// It implements ISalePosition and IPresentView interfaces.
	/// </summary>
	/// <TgData>
	/// </TgData>
    public class SalePosition : ISalePosition, IPresentView
    {

		//Codes of linear attributes for IPresentView.
		/// <summary>
		/// Item code.
		/// </summary>
		public const int ATTR_ITEM = 0;
		/// <summary>
		/// Name code.
		/// </summary>
		public const int ATTR_NAME = 1;
		/// <summary>
		/// Unit code.
		/// </summary>
		public const int ATTR_UNIT = 2;
		/// <summary>
		/// Quantity code.
		/// </summary>
		public const int ATTR_QUANTITY = 3;
		/// <summary>
		/// Price code.
		/// </summary>
		public const int ATTR_PRICE = 4;

		private const int ATTR_COUNT = 5;
		
		/// <summary>
		/// A reference to the product.
		/// </summary>
		private IProduct m_objProduct;

		/// <summary>
		/// A number of product items.
		/// </summary>
		private int m_intQuantity;

		/// <summary>
		/// Constructor.
		/// </summary>
		/// <param name="objProduct"> A reference to the product </param>
		/// <param name="intQuantity"> A number of product items </param>
        public SalePosition( IProduct objProduct, int intQuantity )
        {
            m_objProduct = objProduct;
			m_intQuantity = intQuantity;
        }

		/// <summary>
		/// Gets a product linked with this position.
		/// </summary>
		public IProduct getProduct() 
		{
			return m_objProduct;
		}

		/// <summary>
		/// Returns a number of product items.
		/// </summary>
		public int getQuantity() 
		{
			return m_intQuantity;
		}

		/// <summary>
		/// Returns an amount of the position.
		/// </summary>
		public Decimal getAmount() 
		{
			return m_objProduct.getPricePerItem() * m_intQuantity;
		}

		/// <summary>
		/// Indicates whether the object has linear presentable attributes.
		/// </summary>
		public bool hasLinearAttributes() 
		{
			return true;
		}

		/// <summary>
		/// Indicates whether the object has table presentable attributes.
		/// </summary>
		public bool hasTableAttributes() 
		{
			return false;
		}

		/// <summary>
		/// Returns the number of linear presentable attributes.
		/// </summary>
		public int getLinearAttrCount() 
		{
			return ATTR_COUNT;
		}

		/// <summary>
		/// Returns the name of a linear presentable attribute.
		/// </summary>
		/// <param name="attr"> The attribute code </param>
		public String getLinearAttrName( int attr ) 
		{
			switch(attr)
			{
				case ATTR_ITEM:
					return "Item";
				case ATTR_NAME:
					return "Name";
				case ATTR_UNIT:
					return "Unit";
				case ATTR_QUANTITY:
					return "Quantity";
				case ATTR_PRICE:
					return "Price";
				default:
					return null;
			}
		}

		/// <summary>
		/// Returns the value of a linear presentable attribute.
		/// </summary>
		/// <param name="attr"> The attribute code </param>
		public String getLinearAttrValue( int attr ) 
		{
			switch(attr)
			{
				case ATTR_ITEM:
					//return Int32.ToString(m_objProduct.getID());
					return Utils.Int2Str(m_objProduct.getID());
				case ATTR_NAME:
					return m_objProduct.getName();
				case ATTR_UNIT:
					return m_objProduct.getUnitName();
				case ATTR_QUANTITY:
					//return Int32.ToString(m_intQuantity);
					return Utils.Int2Str(m_intQuantity);
				case ATTR_PRICE:
					//return Decimal.ToString(m_objProduct.getPricePerItem() * m_intQuantity);
					return Utils.AsCurrency(m_objProduct.getPricePerItem() * m_intQuantity);
				default:
					return null;
			}
		}

		/// <summary>
		/// Returns the number of table presentable attributes.
		/// </summary>
		public int getTableAttrCount() 
		{
			return 0;	
		}

		/// <summary>
		/// Returns the size of attribute table.
		/// </summary>
		public int getTableRowCount() 
		{
			return 0;	
		}

		/// <summary>
		/// Returns the name of a table presentable attribute.
		/// </summary>
		/// <param name="attr"> The attribute code </param>
		public String getTableAttrName( int attr ) 
		{
			return null;	
		}

		/// <summary>
		/// Returns the value of a table presentable attribute.
		/// </summary>
		/// <param name="attr"> The attribute code </param>
		/// <param name="row"> The table row </param>
		public String getTableAttrValue( int attr, int row ) 
		{
			return null;			
		}

    }
}
